#!/usr/bin/env python3
import argparse, csv, collections, math, os

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--csv",
        default=r"data\results\vol4_wilson_loop_pipeline_crossover_analysis\crossover_summary.csv",
        help="Path to crossover_summary.csv")
    ap.add_argument("--spread", type=float, default=0.15,
        help="Min relative spread threshold (default 0.15 = 15%)")
    ap.add_argument("--cv", type=float, default=0.03,
        help="Min coefficient-of-variation threshold (default 0.03 = 3%)")
    args = ap.parse_args()

    path = os.path.normpath(args.csv)
    if not os.path.exists(path):
        raise SystemExit(f"ERROR: file not found: {path}")

    rows = list(csv.DictReader(open(path, encoding="utf-8")))
    by = collections.defaultdict(list)
    for r in rows:
        L = int(r["L"])
        g = r.get("gauge_group") or r.get("gauge") or "?"
        sc = float(r["sigma_c"])
        by[(L, g)].append(sc)

    flagged = 0
    for key, vals in sorted(by.items()):
        if len(vals) < 2:
            continue
        mu = sum(vals) / len(vals)
        spread = max(vals) - min(vals)
        cv = (sum((v - mu) ** 2 for v in vals) / len(vals)) ** 0.5 / (mu + 1e-12)
        if spread < args.spread * max(mu, 1e-12) or cv < args.cv:
            print(f"LOW VAR  L={key[0]:>2}  gauge={key[1]:<3}  "
                  f"N={len(vals)}  spread≈{spread:.4g}  cv≈{cv:.4g}")
            flagged += 1

    if not flagged:
        print("Looks good: no suspiciously low variance across b/k/n0.")

if __name__ == "__main__":
    main()
